function getToken() {
  return localStorage.getItem("sc_token") || "";
}

async function fetchJSON(url, fallback) {
  try {
    const res = await fetch(url, {
      headers: { "Authorization": "Bearer " + getToken() }
    });
    if (!res.ok) throw new Error("Bad status");
    return await res.json();
  } catch (e) {
    console.warn("Using fallback for", url);
    return fallback;
  }
}

function fillTable(tableId, rows) {
  const tbody = document.querySelector(`#${tableId} tbody`);
  if (!tbody) return;
  tbody.innerHTML = "";
  rows.forEach(r => {
    const tr = document.createElement("tr");
    tr.innerHTML = r.map(col => `<td>${col}</td>`).join("");
    tbody.appendChild(tr);
  });
}

async function initSuperadminDashboard() {
  const data = await fetchJSON("/api/superadmin/dashboard", {
    totalUsers: 1520,
    totalAgents: 38,
    liveUsers: 214,
    liveAgents: 12,
    totalBalance: 3420000,
    runningBalance: 1280000,
    totalWithdrawals: 840000,
    totalGameAmount: 5600000,
    profitLoss: "+ 3,20,000",
    pendingWithdrawals: 23
  });

  const stats = [
    { label: "Total Users", value: data.totalUsers, sub: "All time" },
    { label: "Total Agents", value: data.totalAgents, sub: "Active + inactive" },
    { label: "Live Users", value: data.liveUsers, sub: "Online now", live: true },
    { label: "Live Agents", value: data.liveAgents, sub: "Online now", live: true },
    { label: "Total Balance", value: data.totalBalance, sub: "Coins in system" },
    { label: "Running Balance", value: data.runningBalance, sub: "In active play" },
    { label: "Total Withdrawals", value: data.totalWithdrawals, sub: "Paid out" },
    { label: "Profit / Loss", value: data.profitLoss, sub: "Overall" }
  ];

  const grid = document.getElementById("statsGridSuperadmin");
  stats.forEach(s => {
    const card = document.createElement("div");
    card.className = "stat-card" + (s.live ? " live" : "");
    card.innerHTML = `
      <div class="stat-label">${s.label}</div>
      <div class="stat-value">${s.value}</div>
      <div class="stat-sub">${s.sub}</div>
    `;
    grid.appendChild(card);
  });

  fillTable("warningTable", [
    ["user_1021", "Multi login", "2 min ago"],
    ["agent_09", "Suspicious IP", "7 min ago"],
    ["user_334", "Too many failed logins", "15 min ago"]
  ]);
}

async function initAgentDashboard() {
  const data = await fetchJSON("/api/agent/dashboard", {
    totalUsers: 120,
    liveUsers: 18,
    totalBalance: 42000,
    pendingWithdrawals: 5,
    approvedWithdrawals: 38,
    profitLoss: "+ 18,500"
  });

  const statsGrid = document.getElementById("statsGridAgent");
  const items = [
    { label: "My Users", value: data.totalUsers, sub: "Total under you" },
    { label: "Live Users", value: data.liveUsers, sub: "Right now", live: true },
    { label: "My Balance", value: data.totalBalance, sub: "Coins available" },
    { label: "Profit / Loss", value: data.profitLoss, sub: "Overall" },
    { label: "Pending Withdrawals", value: data.pendingWithdrawals, sub: "Need action" },
    { label: "Approved Withdrawals", value: data.approvedWithdrawals, sub: "Completed" }
  ];

  items.forEach(s => {
    const card = document.createElement("div");
    card.className = "stat-card" + (s.live ? " live" : "");
    card.innerHTML = `
      <div class="stat-label">${s.label}</div>
      <div class="stat-value">${s.value}</div>
      <div class="stat-sub">${s.sub}</div>
    `;
    statsGrid.appendChild(card);
  });

  fillTable("agentUserTable", [
    ["user_101", "53", "₹ 12,300", "Active"],
    ["user_145", "31", "₹ 7,900", "Active"],
    ["user_173", "9", "₹ 1,200", "Idle"]
  ]);

  fillTable("agentWithdrawals", [
    ["user_145", "₹ 3,000", "10 min ago"],
    ["user_173", "₹ 800", "25 min ago"]
  ]);
}

async function initUserDashboard() {
  const data = await fetchJSON("/api/user/dashboard", {
    wallet: 1800,
    gamesToday: 14,
    totalWagered: 5400,
    totalWon: 3200,
    pendingWithdrawals: 1
  });

  const pill = document.getElementById("userWalletPill");
  if (pill) pill.textContent = "Wallet: " + data.wallet;

  const grid = document.getElementById("statsGridUser");
  const items = [
    { label: "My Wallet", value: data.wallet, sub: "Available coins", live: true },
    { label: "Games Today", value: data.gamesToday, sub: "So far" },
    { label: "Total Wagered", value: data.totalWagered, sub: "Coins bet" },
    { label: "Total Won", value: data.totalWon, sub: "Coins won" },
    { label: "Pending Withdrawals", value: data.pendingWithdrawals, sub: "Waiting approval" }
  ];

  items.forEach(s => {
    const card = document.createElement("div");
    card.className = "stat-card" + (s.live ? " live" : "");
    card.innerHTML = `
      <div class="stat-label">${s.label}</div>
      <div class="stat-value">${s.value}</div>
      <div class="stat-sub">${s.sub}</div>
    `;
    grid.appendChild(card);
  });

  fillTable("userGamesTable", [
    ["Color Wheel", "200", "+400", "5 min ago"],
    ["Color Wheel", "100", "-100", "12 min ago"],
    ["Dice", "150", "+0", "28 min ago"]
  ]);

  fillTable("userWithdrawalsTable", [
    ["₹ 2,000", "Pending", "Today 11:21"],
    ["₹ 1,000", "Approved", "Yesterday"]
  ]);
}

document.addEventListener("DOMContentLoaded", () => {
  const main = document.querySelector(".app-main");
  if (!main) return;
  const role = main.dataset.role;

  if (role === "superadmin") initSuperadminDashboard();
  if (role === "agent") initAgentDashboard();
  if (role === "user") initUserDashboard();

  document.querySelectorAll(".app-nav button").forEach(btn => {
    btn.addEventListener("click", () => {
      const group = btn.closest(".app-sidebar").querySelectorAll(".app-nav button");
      group.forEach(b => b.classList.remove("active"));
      btn.classList.add("active");
    });
  });
});
