// === V10: BULK USER CREATE (CSV) ===
//
// Paste this into backend/src/routes/superadmin.js BEFORE module.exports = router;

router.post("/users/bulk-create", async (req, res) => {
  const { csv } = req.body;
  if (!csv) return res.status(400).json({ message: "CSV data required" });

  const lines = csv.split(/\r?\n/).map(l => l.trim()).filter(Boolean);
  if (!lines.length) return res.status(400).json({ message: "No rows" });

  // expected CSV header:
  // username,password,role,agent_id
  const header = lines[0].toLowerCase().split(",");
  const idxUser = header.indexOf("username");
  const idxPass = header.indexOf("password");
  const idxRole = header.indexOf("role");
  const idxAgent = header.indexOf("agent_id");

  if (idxUser === -1 || idxPass === -1) {
    return res.status(400).json({ message: "CSV must have username,password,[role],[agent_id] columns" });
  }

  const created = [];
  const bcrypt = require("bcryptjs");
  let conn;

  try {
    conn = await pool.getConnection();
    await conn.beginTransaction();

    for (let i = 1; i < lines.length; i++) {
      const parts = lines[i].split(",");
      if (parts.length < 2) continue;
      const username = parts[idxUser].trim();
      const password = parts[idxPass].trim() || "123456";
      const roleStr = idxRole !== -1 ? parts[idxRole].trim().toLowerCase() : "user";
      const agentIdRaw = idxAgent !== -1 ? parts[idxAgent].trim() : "";
      if (!username) continue;

      let roleId = 3; // default user
      if (roleStr === "agent") roleId = 2;
      if (roleStr === "superadmin") roleId = 1;
      if (roleStr === "developer") roleId = 4;

      let agentId = null;
      if (agentIdRaw) {
        const num = Number(agentIdRaw);
        if (!Number.isNaN(num)) agentId = num;
      }

      const hash = await bcrypt.hash(password, 10);
      const [ins] = await conn.query(
        "INSERT INTO users (username, password_hash, role_id, agent_id) VALUES (?,?,?,?)",
        [username, hash, roleId, agentId]
      );
      const newId = ins.insertId;

      await conn.query(
        "INSERT INTO wallets (user_id, balance) VALUES (?, 0)",
        [newId]
      );

      created.push({ id: newId, username, role_id: roleId, agent_id: agentId || null, plain_password: password });
    }

    await conn.commit();
    res.json({ success: true, created });
  } catch (err) {
    if (conn) {
      try { await conn.rollback(); } catch {}
    }
    console.error("bulk-create users error:", err);
    res.status(500).json({ message: "Internal server error" });
  } finally {
    if (conn) conn.release();
  }
});
