const express = require("express");
const pool = require("../config/db");
const { requireRole } = require("../middleware/auth");
const router = express.Router();

/**
 * User creates withdrawal request.
 * We immediately lock coins by deducting from the user's wallet so they
 * cannot spend the same coins again. On rejection, we refund.
 */
router.post("/request", async (req, res) => {
  const userId = req.user.id;
  const { amount } = req.body;
  const amt = Number(amount);
  if (!amt || amt <= 0) {
    return res.status(400).json({ message: "Invalid amount" });
  }

  let conn;
  try {
    conn = await pool.getConnection();
    await conn.beginTransaction();

    const [[wallet]] = await conn.query(
      "SELECT id, balance FROM wallets WHERE user_id = ? FOR UPDATE",
      [userId]
    );
    if (!wallet || wallet.balance < amt) {
      await conn.rollback();
      return res.status(400).json({ message: "Insufficient balance" });
    }

    await conn.query(
      "UPDATE wallets SET balance = balance - ? WHERE id = ?",
      [amt, wallet.id]
    );

    const [ins] = await conn.query(
      "INSERT INTO withdrawals (user_id, amount, status) VALUES (?, ?, 'pending')",
      [userId, amt]
    );
    const withdrawalId = ins.insertId;

    await conn.query(
      `INSERT INTO coin_transactions (from_user_id, to_user_id, amount, type, metadata)
       VALUES (?, NULL, ?, 'withdraw_lock',
               JSON_OBJECT('withdrawal_id', ?, 'note','user requested withdrawal'))`,
      [userId, amt, withdrawalId]
    );

    await conn.commit();
    res.json({ success: true, withdrawalId, remainingBalance: wallet.balance - amt });
  } catch (err) {
    if (conn) {
      try { await conn.rollback(); } catch {}
    }
    console.error("withdraw request error:", err);
    res.status(500).json({ message: "Internal server error" });
  } finally {
    if (conn) conn.release();
  }
});

/**
 * List withdrawals:
 * - superadmin: all
 * - agent: only for their users
 * - user: only self
 */
router.get("/list", async (req, res) => {
  const role = req.user.role;
  const userId = req.user.id;

  try {
    let rows;
    if (role === "superadmin") {
      const [r] = await pool.query(
        `SELECT w.*, u.username
         FROM withdrawals w
         JOIN users u ON u.id = w.user_id
         ORDER BY w.created_at DESC
         LIMIT 200`
      );
      rows = r;
    } else if (role === "agent") {
      const [r] = await pool.query(
        `SELECT w.*, u.username
         FROM withdrawals w
         JOIN users u ON u.id = w.user_id
         WHERE u.agent_id = ?
         ORDER BY w.created_at DESC
         LIMIT 200`,
        [userId]
      );
      rows = r;
    } else {
      const [r] = await pool.query(
        `SELECT w.*, u.username
         FROM withdrawals w
         JOIN users u ON u.id = w.user_id
         WHERE w.user_id = ?
         ORDER BY w.created_at DESC
         LIMIT 50`,
        [userId]
      );
      rows = r;
    }
    res.json(rows);
  } catch (err) {
    console.error("withdrawals list error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

/**
 * Approve / reject by superadmin.
 * - approved: coins stay deducted, we only log transaction as finalized payout
 * - rejected: coins are refunded back to user's wallet and logged
 */
router.post("/decide", requireRole("superadmin"), async (req, res) => {
  const deciderId = req.user.id;
  const { withdrawalId, action } = req.body;
  if (!withdrawalId || !["approved","rejected"].includes(action)) {
    return res.status(400).json({ message: "Invalid action" });
  }

  let conn;
  try {
    conn = await pool.getConnection();
    await conn.beginTransaction();

    const [rows] = await conn.query(
      "SELECT * FROM withdrawals WHERE id = ? FOR UPDATE",
      [withdrawalId]
    );
    if (!rows.length) {
      await conn.rollback();
      return res.status(404).json({ message: "Withdrawal not found" });
    }
    const wd = rows[0];
    if (wd.status !== "pending") {
      await conn.rollback();
      return res.status(400).json({ message: "Already decided" });
    }

    const userId = wd.user_id;
    const amt = wd.amount;

    if (action === "approved") {
      await conn.query(
        "UPDATE withdrawals SET status = 'approved', decided_by = ?, decided_at = NOW() WHERE id = ?",
        [deciderId, withdrawalId]
      );

      await conn.query(
        `INSERT INTO coin_transactions (from_user_id, to_user_id, amount, type, metadata)
         VALUES (?, ?, ?, 'withdraw_approved',
                 JSON_OBJECT('withdrawal_id', ?, 'note','payout approved'))`,
        [userId, deciderId, amt, withdrawalId]
      );
    } else {
      const [[wallet]] = await conn.query(
        "SELECT id, balance FROM wallets WHERE user_id = ? FOR UPDATE",
        [userId]
      );
      let walletId = wallet ? wallet.id : null;
      if (!walletId) {
        const [insW] = await conn.query(
          "INSERT INTO wallets (user_id, balance) VALUES (?, 0)",
          [userId]
        );
        walletId = insW.insertId;
      }

      await conn.query(
        "UPDATE wallets SET balance = balance + ? WHERE id = ?",
        [amt, walletId]
      );

      await conn.query(
        "UPDATE withdrawals SET status = 'rejected', decided_by = ?, decided_at = NOW() WHERE id = ?",
        [deciderId, withdrawalId]
      );

      await conn.query(
        `INSERT INTO coin_transactions (from_user_id, to_user_id, amount, type, metadata)
         VALUES (?, ?, ?, 'withdraw_refund',
                 JSON_OBJECT('withdrawal_id', ?, 'note','withdrawal rejected - refund'))`,
        [deciderId, userId, amt, withdrawalId]
      );
    }

    await conn.commit();
    res.json({ success: true });
  } catch (err) {
    if (conn) {
      try { await conn.rollback(); } catch {}
    }
    console.error("decide withdrawal error:", err);
    res.status(500).json({ message: "Internal server error" });
  } finally {
    if (conn) conn.release();
  }
});

module.exports = router;
