const express = require("express");
const pool = require("../config/db");
const router = express.Router();

const wheelSegments = [
  "1","2","3","4","50X","5","6","7","8","9",
  "75X","10","11","12","13","100X","14","15","16","17","18","25X"
];

function getMultiplier(result, guessed) {
  if (result !== guessed) return 0;
  if (result.endsWith("X")) {
    const n = parseInt(result.replace("X",""), 10);
    return isNaN(n) ? 0 : n;
  }
  const num = parseInt(result, 10);
  if (!isNaN(num)) return 2;
  return 0;
}

router.get("/dashboard", async (req, res) => {
  try {
    const userId = req.user.id;
    const [[wallet]] = await pool.query(
      "SELECT balance FROM wallets WHERE user_id = ? LIMIT 1",
      [userId]
    );
    const [wdRows] = await pool.query(
      "SELECT COUNT(*) AS pending FROM withdrawals WHERE user_id = ? AND status = 'pending'",
      [userId]
    );
    res.json({
      wallet: wallet ? wallet.balance : 0,
      gamesToday: 0,
      totalWagered: 0,
      totalWon: 0,
      pendingWithdrawals: wdRows[0].pending
    });
  } catch (err) {
    console.error("User dashboard error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

router.get("/wallet", async (req, res) => {
  const userId = req.user.id;
  try {
    const [rows] = await pool.query(
      "SELECT id, balance FROM wallets WHERE user_id = ? LIMIT 1",
      [userId]
    );
    if (!rows.length) {
      const [r] = await pool.query(
        "INSERT INTO wallets (user_id, balance) VALUES (?, 0)",
        [userId]
      );
      return res.json({ walletId: r.insertId, balance: 0 });
    }
    res.json({ walletId: rows[0].id, balance: rows[0].balance });
  } catch (err) {
    console.error("Wallet error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

router.post("/spin", async (req, res) => {
  const userId = req.user.id;
  const { betAmount, guessedSlot } = req.body;
  const bet = Number(betAmount);

  if (!bet || bet <= 0) return res.status(400).json({ message: "Invalid bet amount" });
  if (!guessedSlot || !wheelSegments.includes(guessedSlot)) {
    return res.status(400).json({ message: "Invalid guessed slot" });
  }

  let conn;
  try {
    conn = await pool.getConnection();
    await conn.beginTransaction();

    const [[settings]] = await conn.query(
      "SELECT agent_commission_percent FROM app_settings WHERE id = 1"
    );

    const [userRows] = await conn.query(
      "SELECT id, agent_id FROM users WHERE id = ? LIMIT 1 FOR UPDATE",
      [userId]
    );
    const currentUser = userRows[0];

    const [walletRows] = await conn.query(
      "SELECT id, balance FROM wallets WHERE user_id = ? FOR UPDATE",
      [userId]
    );
    let walletId, balance;
    if (!walletRows.length) {
      const [ins] = await conn.query(
        "INSERT INTO wallets (user_id, balance) VALUES (?, 0)",
        [userId]
      );
      walletId = ins.insertId;
      balance = 0;
    } else {
      walletId = walletRows[0].id;
      balance = Number(walletRows[0].balance);
    }

    if (balance < bet) {
      await conn.rollback();
      return res.status(400).json({ message: "Insufficient balance", balance });
    }

    const afterBet = balance - bet;
    await conn.query("UPDATE wallets SET balance = ? WHERE id = ?", [afterBet, walletId]);

    const index = Math.floor(Math.random() * wheelSegments.length);
    const resultSlot = wheelSegments[index];
    const mult = getMultiplier(resultSlot, guessedSlot);
    const winAmount = bet * mult;
    const finalBalance = afterBet + winAmount;

    await conn.query("UPDATE wallets SET balance = ? WHERE id = ?", [finalBalance, walletId]);

    await conn.query(
      `INSERT INTO coin_transactions (from_user_id, to_user_id, amount, type, metadata)
       VALUES (?, NULL, ?, 'game_bet', JSON_OBJECT('game','color_wheel','guessed',?, 'result',?))`,
      [userId, bet, guessedSlot, resultSlot]
    );

    let roundId = null;
    const [roundIns] = await conn.query(
      `INSERT INTO game_rounds (game_id, user_id, bet_amount, win_amount, result_details)
       VALUES (1, ?, ?, ?, JSON_OBJECT('slot', ?, 'guessed', ?))`,
      [userId, bet, winAmount, resultSlot, guessedSlot]
    );
    roundId = roundIns.insertId;

    if (winAmount > 0) {
      await conn.query(
        `INSERT INTO coin_transactions (from_user_id, to_user_id, amount, type, metadata)
         VALUES (NULL, ?, ?, 'game_win', JSON_OBJECT('game','color_wheel','guessed',?, 'result',?))`,
        [userId, winAmount, guessedSlot, resultSlot]
      );
    }

    // Agent commission (on house profit)
    const houseProfit = bet - winAmount;
    if (houseProfit > 0 && currentUser.agent_id) {
      const percent = settings ? (settings.agent_commission_percent || 0) : 0;
      const commission = Math.floor(houseProfit * percent / 100);
      if (commission > 0) {
        // credit to agent wallet
        const [agentWalletRows] = await conn.query(
          "SELECT id FROM wallets WHERE user_id = ? FOR UPDATE",
          [currentUser.agent_id]
        );
        if (agentWalletRows.length) {
          await conn.query(
            "UPDATE wallets SET balance = balance + ? WHERE id = ?",
            [commission, agentWalletRows[0].id]
          );
        } else {
          const [aw] = await conn.query(
            "INSERT INTO wallets (user_id, balance) VALUES (?, ?)",
            [currentUser.agent_id, commission]
          );
        }
        await conn.query(
          `INSERT INTO agent_commissions (agent_id, user_id, game_round_id, commission_amount)
           VALUES (?, ?, ?, ?)`,
          [currentUser.agent_id, userId, roundId, commission]
        );
      }
    }

    await conn.commit();

    res.json({
      success: true,
      segmentIndex: index,
      resultSlot,
      betAmount: bet,
      winAmount,
      finalBalance
    });
  } catch (err) {
    if (conn) {
      try { await conn.rollback(); } catch {}
    }
    console.error("Spin error:", err);
    res.status(500).json({ message: "Internal server error" });
  } finally {
    if (conn) conn.release();
  }
});



// Transactions + withdrawals history for current user
router.get("/transactions", async (req, res) => {
  const userId = req.user.id;
  try {
    const [tx] = await pool.query(
      `SELECT id, from_user_id, to_user_id, amount, type, metadata, created_at
       FROM coin_transactions
       WHERE from_user_id = ? OR to_user_id = ?
       ORDER BY created_at DESC
       LIMIT 100`,
      [userId, userId]
    );
    const [wd] = await pool.query(
      `SELECT id, amount, status, created_at
       FROM withdrawals
       WHERE user_id = ?
       ORDER BY created_at DESC
       LIMIT 50`,
      [userId]
    );
    res.json({ transactions: tx, withdrawals: wd });
  } catch (err) {
    console.error("user transactions error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

module.exports = router;
