const express = require("express");
const pool = require("../config/db");
const router = express.Router();

router.get("/dashboard", async (req, res) => {
  try {
    const [[userAgg]] = await pool.query("SELECT COUNT(*) AS totalUsers FROM users WHERE role_id = 3");
    const [[agentAgg]] = await pool.query("SELECT COUNT(*) AS totalAgents FROM users WHERE role_id = 2");
    const [[balanceAgg]] = await pool.query("SELECT SUM(balance) AS totalBalance FROM wallets");
    const [[wdAgg]] = await pool.query("SELECT SUM(amount) AS totalWithdrawals FROM withdrawals WHERE status='approved'");
    const [[roundAgg]] = await pool.query("SELECT SUM(bet_amount) AS totalGameAmount FROM game_rounds");

    res.json({
      totalUsers: userAgg.totalUsers || 0,
      totalAgents: agentAgg.totalAgents || 0,
      liveUsers: 0,
      liveAgents: 0,
      totalBalance: balanceAgg.totalBalance || 0,
      runningBalance: 0,
      totalWithdrawals: wdAgg.totalWithdrawals || 0,
      totalGameAmount: roundAgg.totalGameAmount || 0,
      profitLoss: 0
    });
  } catch (err) {
    console.error("superadmin dashboard error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

router.get("/analytics", async (req, res) => {
  try {
    const [rows] = await pool.query(`
      SELECT DATE(created_at) AS day,
             SUM(bet_amount) AS total_bet,
             SUM(win_amount) AS total_win
      FROM game_rounds
      GROUP BY DATE(created_at)
      ORDER BY day DESC
      LIMIT 7
    `);
    res.json(rows);
  } catch (err) {
    console.error("analytics error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// Extended analytics: profit per day + agent commissions + top users
router.get("/analytics-extended", async (req, res) => {
  try {
    const [days] = await pool.query(`
      SELECT DATE(created_at) AS day,
             SUM(bet_amount) AS total_bet,
             SUM(win_amount) AS total_win,
             SUM(bet_amount - win_amount) AS profit
      FROM game_rounds
      GROUP BY DATE(created_at)
      ORDER BY day DESC
      LIMIT 14
    `);

    const [agentPerf] = await pool.query(`
      SELECT u.id, u.username, COALESCE(SUM(a.commission_amount),0) AS total_commission
      FROM users u
      LEFT JOIN agent_commissions a ON a.agent_id = u.id
      WHERE u.role_id = 2
      GROUP BY u.id, u.username
      ORDER BY total_commission DESC
      LIMIT 10
    `);

    const [topUsers] = await pool.query(`
      SELECT u.id, u.username,
             COALESCE(SUM(g.bet_amount),0) AS total_bet,
             COALESCE(SUM(g.win_amount),0) AS total_win
      FROM users u
      LEFT JOIN game_rounds g ON g.user_id = u.id
      WHERE u.role_id = 3
      GROUP BY u.id, u.username
      ORDER BY total_bet DESC
      LIMIT 10
    `);

    res.json({ days, agentPerf, topUsers });
  } catch (err) {
    console.error("analytics-extended error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// Live users map data (approx)
// Uses active sessions + last login_logs ip; coordinates are placeholder
router.get("/live-users", async (req, res) => {
  try {
    const [rows] = await pool.query(`
      SELECT s.user_id, u.username, s.ip
      FROM sessions s
      JOIN users u ON u.id = s.user_id
      WHERE s.is_active = 1
      ORDER BY s.created_at DESC
      LIMIT 100
    `);

    // map IP -> pseudo coordinates (for demo).
    const mapped = rows.map(r => {
      let lat = 20.0, lng = 78.0; // default India center
      if (r.ip) {
        const parts = String(r.ip).split(".").map(x => parseInt(x, 10) || 0);
        if (parts.length >= 2) {
          lat = -60 + (parts[0] % 120); // -60..60
          lng = -170 + (parts[1] % 340); // -170..170
        }
      }
      return {
        userId: r.user_id,
        username: r.username,
        ip: r.ip,
        lat,
        lng
      };
    });

    res.json(mapped);
  } catch (err) {
    console.error("live-users error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

router.get("/settings", async (req, res) => {
  try {
    const [[s]] = await pool.query("SELECT * FROM app_settings WHERE id = 1");
    res.json(s || {});
  } catch (err) {
    console.error("settings get error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

router.post("/settings", async (req, res) => {
  const {
    maintenance_mode,
    referral_new_user_bonus,
    referral_referrer_bonus,
    agent_commission_percent
  } = req.body;

  try {
    await pool.query(
      `UPDATE app_settings
       SET maintenance_mode = COALESCE(?, maintenance_mode),
           referral_new_user_bonus = COALESCE(?, referral_new_user_bonus),
           referral_referrer_bonus = COALESCE(?, referral_referrer_bonus),
           agent_commission_percent = COALESCE(?, agent_commission_percent),
           updated_at = NOW()
       WHERE id = 1`,
      [
        maintenance_mode !== undefined ? Number(maintenance_mode) : null,
        referral_new_user_bonus !== undefined ? Number(referral_new_user_bonus) : null,
        referral_referrer_bonus !== undefined ? Number(referral_referrer_bonus) : null,
        agent_commission_percent !== undefined ? Number(agent_commission_percent) : null
      ]
    );
    res.json({ success: true });
  } catch (err) {
    console.error("settings update error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});



// List users/agents for management UI
router.get("/users", async (req, res) => {
  try {
    const scope = req.query.scope || "users";
    let where = "";
    if (scope === "users") {
      where = "WHERE u.role_id = 3";
    } else if (scope === "agents") {
      where = "WHERE u.role_id = 2";
    } else if (scope === "agentSelf") {
      // special scope: only users under current agent (for agent dashboard)
      where = "WHERE u.role_id = 3 AND u.agent_id = ?";
    } else {
      where = "";
    }

    let params = [];
    if (scope === "agentSelf") {
      params = [req.user.id];
    }

    const [rows] = await pool.query(
      `
      SELECT u.id, u.role_id, u.username, u.agent_id, u.is_blocked, u.created_at,
             w.balance,
             a.username AS agent_username
      FROM users u
      LEFT JOIN wallets w ON w.user_id = u.id
      LEFT JOIN users a ON a.id = u.agent_id
      ${where}
      ORDER BY u.created_at DESC
      LIMIT 200
      `,
      params
    );
    res.json(rows);
  } catch (err) {
    console.error("superadmin users list error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// Block / unblock user
router.post("/users/block", async (req, res) => {
  const { userId, blocked } = req.body;
  if (!userId) return res.status(400).json({ message: "Missing userId" });
  try {
    await pool.query(
      "UPDATE users SET is_blocked = ? WHERE id = ?",
      [blocked ? 1 : 0, userId]
    );
    res.json({ success: true });
  } catch (err) {
    console.error("superadmin block error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// Set or clear agent for a user
router.post("/users/set-agent", async (req, res) => {
  const { userId, agentId } = req.body;
  if (!userId) return res.status(400).json({ message: "Missing userId" });

  try {
    if (agentId) {
      // ensure target is agent
      const [[agent]] = await pool.query(
        "SELECT id FROM users WHERE id = ? AND role_id = 2",
        [agentId]
      );
      if (!agent) {
        return res.status(400).json({ message: "Invalid agentId" });
      }
    }
    await pool.query(
      "UPDATE users SET agent_id = ? WHERE id = ?",
      [agentId || null, userId]
    );
    res.json({ success: true });
  } catch (err) {
    console.error("superadmin set-agent error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});



// Global coin transactions overview
router.get("/transactions", async (req, res) => {
  try {
    const [rows] = await pool.query(
      `SELECT c.id,
              c.from_user_id,
              fu.username AS from_username,
              c.to_user_id,
              tu.username AS to_username,
              c.amount,
              c.type,
              c.metadata,
              c.created_at
       FROM coin_transactions c
       LEFT JOIN users fu ON fu.id = c.from_user_id
       LEFT JOIN users tu ON tu.id = c.to_user_id
       ORDER BY c.created_at DESC
       LIMIT 200`
    );
    res.json(rows);
  } catch (err) {
    console.error("superadmin transactions error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});



// Security overview: active sessions & login logs
router.get("/security/sessions", async (req, res) => {
  try {
    const [rows] = await pool.query(
      `SELECT s.id, s.user_id, u.username, s.ip, s.user_agent, s.is_active, s.created_at
       FROM sessions s
       JOIN users u ON u.id = s.user_id
       ORDER BY s.created_at DESC
       LIMIT 200`
    );
    res.json(rows);
  } catch (err) {
    console.error("superadmin security sessions error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

router.get("/security/logins", async (req, res) => {
  try {
    const [rows] = await pool.query(
      `SELECT l.id, l.user_id, u.username, l.ip, l.user_agent, l.success, l.created_at
       FROM login_logs l
       LEFT JOIN users u ON u.id = l.user_id
       ORDER BY l.created_at DESC
       LIMIT 200`
    );
    res.json(rows);
  } catch (err) {
    console.error("superadmin security logins error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

module.exports = router;
