const express = require("express");
const crypto = require("crypto");
const pool = require("../config/db");

const router = express.Router();

/**
 * NOTE:
 * - This file contains REALISTIC server-side skeletons for Razorpay & Paytm.
 * - You MUST install and configure the respective SDKs / checksum utilities
 *   and set the required ENV variables in your backend .env:
 *
 *   # Razorpay
 *   RAZORPAY_KEY_ID=your_key
 *   RAZORPAY_KEY_SECRET=your_secret
 *
 *   # Paytm
 *   PAYTM_MID=your_mid
 *   PAYTM_MERCHANT_KEY=your_merchant_key
 *   PAYTM_WEBSITE=WEBSTAGING or DEFAULT
 *   PAYTM_CHANNEL_ID=WEB
 *   PAYTM_INDUSTRY_TYPE_ID=Retail
 *   PAYTM_CALLBACK_URL=https://yourdomain.com/api/payments/paytm/callback
 */

// -------- Utility: credit coins on successful payment --------
async function creditCoins(userId, coins, reference) {
  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();
    const [[wallet]] = await conn.query(
      "SELECT id, balance FROM wallets WHERE user_id = ? FOR UPDATE",
      [userId]
    );
    let walletId = wallet ? wallet.id : null;
    if (!walletId) {
      const [ins] = await conn.query(
        "INSERT INTO wallets (user_id, balance) VALUES (?, 0)",
        [userId]
      );
      walletId = ins.insertId;
    }
    await conn.query(
      "UPDATE wallets SET balance = balance + ? WHERE id = ?",
      [coins, walletId]
    );
    await conn.query(
      "INSERT INTO coin_transactions (from_user_id, to_user_id, amount, type, metadata) VALUES (?, ?, ?, 'deposit', JSON_OBJECT('ref', ?))",
      [null, userId, coins, reference || ""]
    );
    await conn.commit();
  } catch (e) {
    await conn.rollback();
    throw e;
  } finally {
    conn.release();
  }
}

// ============ Razorpay Integration ============
// You should `npm install razorpay` in your backend project.
let Razorpay;
try {
  Razorpay = require("razorpay");
} catch (e) {
  console.warn("razorpay package not installed; Razorpay integration will not work until you install it.");
}

/**
 * Create Razorpay order.
 * Frontend will use this to open Razorpay Checkout.
 */
router.post("/razorpay/order", async (req, res) => {
  const userId = req.user.id;
  const { amount } = req.body; // rupees

  const rupees = Number(amount);
  if (!rupees || rupees <= 0) {
    return res.status(400).json({ message: "Invalid amount" });
  }

  if (!Razorpay) {
    return res.status(500).json({ message: "Razorpay SDK not installed on server" });
  }

  try {
    const instance = new Razorpay({
      key_id: process.env.RAZORPAY_KEY_ID,
      key_secret: process.env.RAZORPAY_KEY_SECRET,
    });

    const options = {
      amount: Math.round(rupees * 100), // to paise
      currency: "INR",
      receipt: "SCAPP_" + userId + "_" + Date.now(),
      notes: {
        userId: String(userId),
        purpose: "SelectColorAppCoins",
      },
    };

    const order = await instance.orders.create(options);
    res.json({
      orderId: order.id,
      amount: order.amount,
      currency: order.currency,
      key: process.env.RAZORPAY_KEY_ID,
    });
  } catch (err) {
    console.error("razorpay order error:", err);
    res.status(500).json({ message: "Failed to create Razorpay order" });
  }
});

/**
 * Verify Razorpay payment signature & credit coins.
 * Frontend should call this AFTER successful Razorpay checkout.
 */
router.post("/razorpay/verify", async (req, res) => {
  const userId = req.user.id;
  const { razorpay_order_id, razorpay_payment_id, razorpay_signature, amount } = req.body;

  if (!razorpay_order_id || !razorpay_payment_id || !razorpay_signature) {
    return res.status(400).json({ message: "Missing Razorpay fields" });
  }

  const body = razorpay_order_id + "|" + razorpay_payment_id;
  const expected = crypto
    .createHmac("sha256", process.env.RAZORPAY_KEY_SECRET || "")
    .update(body.toString())
    .digest("hex");

  if (expected !== razorpay_signature) {
    return res.status(400).json({ message: "Invalid signature" });
  }

  // Signature valid → credit coins.
  // Simple mapping: 1 rupee = 1 coin (you can adjust)
  const rupees = Number(amount);
  const coins = Math.round(rupees);

  try {
    await creditCoins(userId, coins, razorpay_payment_id);
    res.json({ success: true, coins });
  } catch (err) {
    console.error("credit coins error:", err);
    res.status(500).json({ message: "Failed to credit coins" });
  }
});

// ============ Paytm Integration (skeleton) ============
// For real Paytm, use official checksum utility:
// https://developer.paytm.com/docs/checksum/
let PaytmChecksum;
try {
  PaytmChecksum = require("paytmchecksum");
} catch (e) {
  console.warn("paytmchecksum package not installed; Paytm integration is skeleton only.");
}

/**
 * Generate Paytm payment parameters & checksum.
 * Frontend will redirect user to Paytm PG with these params.
 */
router.post("/paytm/order", async (req, res) => {
  const userId = req.user.id;
  const { amount } = req.body;
  const rupees = Number(amount);

  if (!rupees || rupees <= 0) {
    return res.status(400).json({ message: "Invalid amount" });
  }
  if (!PaytmChecksum) {
    return res.status(500).json({ message: "Paytm checksum utility not installed" });
  }

  const orderId = "SCAPP_" + userId + "_" + Date.now();
  const paytmParams = {
    MID: process.env.PAYTM_MID,
    WEBSITE: process.env.PAYTM_WEBSITE || "WEBSTAGING",
    CHANNEL_ID: process.env.PAYTM_CHANNEL_ID || "WEB",
    INDUSTRY_TYPE_ID: process.env.PAYTM_INDUSTRY_TYPE_ID || "Retail",
    ORDER_ID: orderId,
    CUST_ID: String(userId),
    TXN_AMOUNT: rupees.toFixed(2),
    CALLBACK_URL: process.env.PAYTM_CALLBACK_URL,
  };

  try {
    const checksum = await PaytmChecksum.generateSignature(
      paytmParams,
      process.env.PAYTM_MERCHANT_KEY
    );
    res.json({ paytmParams, checksum });
  } catch (err) {
    console.error("paytm checksum error:", err);
    res.status(500).json({ message: "Failed to generate Paytm checksum" });
  }
});

/**
 * Paytm callback endpoint.
 * Configure this URL in PAYTM_CALLBACK_URL.
 * Here, you must:
 *   - verify checksum
 *   - verify transaction status with Paytm server
 *   - then credit coins
 */
router.post("/paytm/callback", express.urlencoded({ extended: false }), async (req, res) => {
  const body = req.body || {};

  try {
    const paytmChecksum = body.CHECKSUMHASH;
    delete body.CHECKSUMHASH;

    const isValid = PaytmChecksum.verifySignature(
      body,
      process.env.PAYTM_MERCHANT_KEY,
      paytmChecksum
    );
    if (!isValid) {
      return res.status(400).send("Checksum mismatch");
    }

    // At this point you should verify transaction status
    // via Paytm "transaction status" API (server-to-server).
    // Skeleton: assume success if STATUS === 'TXN_SUCCESS'

    if (body.STATUS !== "TXN_SUCCESS") {
      return res.status(400).send("Payment not successful");
    }

    const userId = Number(body.CUST_ID);
    const rupees = Number(body.TXNAMOUNT);
    const coins = Math.round(rupees);

    await creditCoins(userId, coins, body.TXNID);

    // You can redirect user to a success page
    res.send("Payment success, coins credited.");
  } catch (err) {
    console.error("paytm callback error:", err);
    res.status(500).send("Internal server error");
  }
});

module.exports = router;
