const express = require("express");
const pool = require("../config/db");
const router = express.Router();

/**
 * Get current plan status for the platform.
 * Only for developer role.
 */
router.get("/plan", async (req, res) => {
  try {
    const [[row]] = await pool.query(
      "SELECT maintenance_mode, plan_expires_at FROM app_settings WHERE id = 1"
    );
    const now = new Date();
    let expiresAt = row ? row.plan_expires_at : null;
    const expired = !expiresAt || new Date(expiresAt) < now;
    res.json({
      now,
      plan_expires_at: expiresAt,
      expired,
      maintenance_mode: row ? row.maintenance_mode : 0
    });
  } catch (err) {
    console.error("developer plan get error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

/**
 * Extend plan by given days.
 * In real production, call this only AFTER verifying payment from gateway.
 */
router.post("/extend", async (req, res) => {
  const { days } = req.body;
  const d = Number(days);
  if (!d || d <= 0) {
    return res.status(400).json({ message: "Invalid days" });
  }

  try {
    await pool.query(
      `UPDATE app_settings
       SET plan_expires_at = CASE
         WHEN plan_expires_at IS NULL OR plan_expires_at < NOW()
           THEN DATE_ADD(NOW(), INTERVAL ? DAY)
         ELSE DATE_ADD(plan_expires_at, INTERVAL ? DAY)
       END,
       maintenance_mode = 0,
       updated_at = NOW()
       WHERE id = 1`,
      [d, d]
    );
    res.json({ success: true });
  } catch (err) {
    console.error("developer extend error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

module.exports = router;
