const express = require("express");
const pool = require("../config/db");
const router = express.Router();

/**
 * Helper to check if user can see a conversation
 */
async function canAccessConversation(user, convId) {
  const [rows] = await pool.query("SELECT * FROM conversations WHERE id = ?", [convId]);
  if (!rows.length) return { ok: false };
  const c = rows[0];

  if (user.role === "superadmin") {
    return { ok: true, conv: c };
  }

  if (c.type === "user-agent") {
    if (user.role === "user" && c.created_by === user.id) return { ok: true, conv: c };
    if (user.role === "agent") {
      const [uRows] = await pool.query("SELECT agent_id FROM users WHERE id = ?", [c.created_by]);
      if (uRows.length && uRows[0].agent_id === user.id) return { ok: true, conv: c };
    }
  } else if (c.type === "user-superadmin") {
    if (user.role === "user" && c.created_by === user.id) return { ok: true, conv: c };
  } else if (c.type === "agent-superadmin") {
    if (user.role === "agent" && c.created_by === user.id) return { ok: true, conv: c };
  }

  return { ok: false };
}

// Create / open chat for current user
router.post("/open", async (req, res) => {
  const user = req.user;
  const { target } = req.body;

  try {
    if (user.role === "user") {
      if (target === "agent") {
        const [[u]] = await pool.query("SELECT agent_id FROM users WHERE id = ?", [user.id]);
        if (!u || !u.agent_id) {
          return res.status(400).json({ message: "No agent assigned" });
        }
        const [rows] = await pool.query(
          "SELECT id FROM conversations WHERE type='user-agent' AND created_by = ? LIMIT 1",
          [user.id]
        );
        if (rows.length) return res.json({ conversationId: rows[0].id });

        const [ins] = await pool.query(
          "INSERT INTO conversations (created_by, type) VALUES (?, 'user-agent')",
          [user.id]
        );
        return res.json({ conversationId: ins.insertId });
      } else if (target === "superadmin") {
        const [rows] = await pool.query(
          "SELECT id FROM conversations WHERE type='user-superadmin' AND created_by = ? LIMIT 1",
          [user.id]
        );
        if (rows.length) return res.json({ conversationId: rows[0].id });

        const [ins] = await pool.query(
          "INSERT INTO conversations (created_by, type) VALUES (?, 'user-superadmin')",
          [user.id]
        );
        return res.json({ conversationId: ins.insertId });
      }
    } else if (user.role === "agent") {
      if (target === "superadmin") {
        const [rows] = await pool.query(
          "SELECT id FROM conversations WHERE type='agent-superadmin' AND created_by = ? LIMIT 1",
          [user.id]
        );
        if (rows.length) return res.json({ conversationId: rows[0].id });

        const [ins] = await pool.query(
          "INSERT INTO conversations (created_by, type) VALUES (?, 'agent-superadmin')",
          [user.id]
        );
        return res.json({ conversationId: ins.insertId });
      }
    }

    return res.status(400).json({ message: "Invalid target for this role" });
  } catch (err) {
    console.error("chat open error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// List conversations for current user
router.get("/my-conversations", async (req, res) => {
  const user = req.user;
  try {
    let rows = [];
    if (user.role === "user") {
      const [r] = await pool.query(
        "SELECT id, type, created_by, created_at FROM conversations WHERE created_by = ? ORDER BY created_at DESC",
        [user.id]
      );
      rows = r;
    } else if (user.role === "agent") {
      const [r] = await pool.query(
        `SELECT c.id, c.type, c.created_by, c.created_at, u.username AS other_name
         FROM conversations c
         JOIN users u ON u.id = c.created_by
         WHERE c.type='user-agent' AND u.agent_id = ?
         UNION ALL
         SELECT c.id, c.type, c.created_by, c.created_at, 'SuperAdmin' AS other_name
         FROM conversations c
         WHERE c.type='agent-superadmin' AND c.created_by = ?
         ORDER BY created_at DESC`,
        [user.id, user.id]
      );
      rows = r;
    } else if (user.role === "superadmin") {
      const [r] = await pool.query(
        `SELECT c.id, c.type, c.created_by, c.created_at, u.username AS other_name
         FROM conversations c
         JOIN users u ON u.id = c.created_by
         WHERE c.type IN ('user-superadmin','agent-superadmin')
         ORDER BY c.created_at DESC
         LIMIT 200`
      );
      rows = r;
    } else {
      rows = [];
    }

    res.json(rows);
  } catch (err) {
    console.error("my-conversations error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// Get messages for conversation
router.get("/messages/:conversationId", async (req, res) => {
  const user = req.user;
  const convId = Number(req.params.conversationId);
  if (!convId) return res.status(400).json({ message: "Invalid conversation id" });

  try {
    const { ok } = await canAccessConversation(user, convId);
    if (!ok) return res.status(403).json({ message: "Forbidden" });

    const [rows] = await pool.query(
      `SELECT m.id, m.sender_id, u.username, m.body, m.created_at
       FROM messages m
       JOIN users u ON u.id = m.sender_id
       WHERE m.conversation_id = ?
       ORDER BY m.created_at ASC, m.id ASC`,
      [convId]
    );

    res.json(rows);
  } catch (err) {
    console.error("get messages error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// Send message
router.post("/message", async (req, res) => {
  const user = req.user;
  const { conversationId, body } = req.body;
  if (!conversationId || !body) return res.status(400).json({ message: "conversationId and body required" });

  try {
    const { ok } = await canAccessConversation(user, conversationId);
    if (!ok) return res.status(403).json({ message: "Forbidden" });

    await pool.query(
      "INSERT INTO messages (conversation_id, sender_id, body) VALUES (?, ?, ?)",
      [conversationId, user.id, body]
    );
    res.json({ success: true });
  } catch (err) {
    console.error("send message error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

module.exports = router;
