const express = require("express");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const pool = require("../config/db");
const nodemailer = require("nodemailer");

const router = express.Router();

async function logLogin(userId, req, success) {
  try {
    const ip = req.headers["x-forwarded-for"] || req.socket.remoteAddress || null;
    const ua = req.headers["user-agent"] || "";
    await pool.query(
      "INSERT INTO login_logs (user_id, ip, user_agent, success) VALUES (?, ?, ?, ?)",
      [userId || null, String(ip).slice(0,64), ua.slice(0,255), success ? 1 : 0]
    );
  } catch (e) {
    console.error("login log error", e.message);
  }
}

function randomJti() {
  return Math.random().toString(36).slice(2) + Date.now().toString(36);
}

async function createSession(user, req) {
  const jti = randomJti();
  const ip = req.headers["x-forwarded-for"] || req.socket.remoteAddress || null;
  const ua = req.headers["user-agent"] || "";
  const fingerprint = (ua || "").slice(0, 200);

  // deactivate old sessions for this user
  await pool.query("UPDATE sessions SET is_active = 0 WHERE user_id = ?", [user.id]);

  await pool.query(
    "INSERT INTO sessions (user_id, token_jti, device_fingerprint, ip, user_agent, is_active) VALUES (?, ?, ?, ?, ?, 1)",
    [user.id, jti, fingerprint, String(ip).slice(0,64), ua.slice(0,255)]
  );

  return jti;
}

function signToken(user, jti) {
  const payload = { id: user.id, username: user.username, role: user.role, jti };
  return jwt.sign(payload, process.env.JWT_SECRET || "super-secret-change-me", { expiresIn: "12h" });
}

// ---------------- Password login ----------------
router.post("/login", async (req, res) => {
  const { username, password } = req.body;
  if (!username || !password) return res.status(400).json({ message: "Username and password required" });

  try {
    const [rows] = await pool.query(
      `SELECT u.id, u.username, u.password_hash, u.is_blocked, r.name AS role
       FROM users u
       JOIN roles r ON r.id = u.role_id
       WHERE u.username = ? LIMIT 1`,
      [username]
    );
    if (!rows.length) {
      await logLogin(null, req, false);
      return res.status(401).json({ message: "Invalid credentials" });
    }

    const user = rows[0];
    if (user.is_blocked) {
      await logLogin(user.id, req, false);
      return res.status(403).json({ message: "User is blocked" });
    }

    const ok = await bcrypt.compare(password, user.password_hash).catch(() => false);
    if (!ok) {
      await logLogin(user.id, req, false);
      return res.status(401).json({ message: "Invalid credentials" });
    }

    const jti = await createSession(user, req);
    const token = signToken(user, jti);
    await logLogin(user.id, req, true);

    res.json({ token, user: { id: user.id, username: user.username, role: user.role } });
  } catch (err) {
    console.error("Login error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// ---------------- Email OTP login ----------------
router.post("/request-otp", async (req, res) => {
  const { email } = req.body;
  if (!email) return res.status(400).json({ message: "Email required" });

  try {
    const [rows] = await pool.query(
      "SELECT u.id, u.username FROM users u WHERE u.email = ? LIMIT 1",
      [email]
    );
    if (!rows.length) return res.status(404).json({ message: "User not found for this email" });
    const user = rows[0];

    const code = String(Math.floor(100000 + Math.random() * 900000));
    const expiresAt = new Date(Date.now() + 10 * 60 * 1000);

    await pool.query(
      "INSERT INTO otp_codes (user_id, email, code, expires_at) VALUES (?, ?, ?, ?)",
      [user.id, email, code, expiresAt]
    );

    try {
      const transporter = nodemailer.createTransport({
        host: process.env.EMAIL_HOST,
        port: Number(process.env.EMAIL_PORT) || 587,
        secure: false,
        auth: {
          user: process.env.EMAIL_USER,
          pass: process.env.EMAIL_PASS
        }
      });

      await transporter.sendMail({
        from: process.env.EMAIL_USER,
        to: email,
        subject: "Your Select Color App OTP",
        text: `Your OTP is: ${code}`,
      });
    } catch (e) {
      console.error("OTP email send failed:", e.message);
    }

    res.json({ success: true });
  } catch (err) {
    console.error("request-otp error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

router.post("/login-otp", async (req, res) => {
  const { email, code } = req.body;
  if (!email || !code) return res.status(400).json({ message: "Email and code are required" });

  try {
    const [rows] = await pool.query(
      `SELECT o.id, o.user_id, o.code, o.expires_at, o.used,
              u.username, u.is_blocked, r.name AS role
       FROM otp_codes o
       JOIN users u ON u.id = o.user_id
       JOIN roles r ON r.id = u.role_id
       WHERE o.email = ? AND o.code = ?
       ORDER BY o.id DESC
       LIMIT 1`,
      [email, code]
    );
    if (!rows.length) return res.status(400).json({ message: "Invalid OTP" });

    const rec = rows[0];
    if (rec.used) return res.status(400).json({ message: "OTP already used" });
    if (new Date(rec.expires_at) < new Date()) return res.status(400).json({ message: "OTP expired" });
    if (rec.is_blocked) return res.status(403).json({ message: "User is blocked" });

    await pool.query("UPDATE otp_codes SET used = 1 WHERE id = ?", [rec.id]);

    const user = { id: rec.user_id, username: rec.username, role: rec.role };
    const jti = await createSession(user, req);
    const token = signToken(user, jti);
    await logLogin(rec.user_id, req, true);

    res.json({ token, user: { id: rec.user_id, username: rec.username, role: rec.role } });
  } catch (err) {
    console.error("login-otp error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// ---------------- Signup with referral (user only) ----------------
router.post("/signup", async (req, res) => {
  const { username, password, email, referralCode } = req.body;
  if (!username || !password) return res.status(400).json({ message: "Username and password required" });

  try {
    const [exists] = await pool.query("SELECT id FROM users WHERE username = ? LIMIT 1", [username]);
    if (exists.length) return res.status(400).json({ message: "Username already taken" });

    const hash = await bcrypt.hash(password, 10);

    let referrer = null;
    if (referralCode) {
      const [rows] = await pool.query(
        "SELECT id FROM users WHERE referral_code = ? LIMIT 1",
        [referralCode]
      );
      if (rows.length) referrer = rows[0];
    }

    const [settingsRows] = await pool.query(
      "SELECT referral_new_user_bonus, referral_referrer_bonus FROM app_settings WHERE id = 1"
    );
    const settings = settingsRows[0] || { referral_new_user_bonus: 0, referral_referrer_bonus: 0 };

    const conn = await pool.getConnection();
    try {
      await conn.beginTransaction();

      const [ins] = await conn.query(
        "INSERT INTO users (role_id, agent_id, username, password_hash, email, referred_by_user_id) VALUES (3, ?, ?, ?, ?, ?)",
        [referrer ? referrer.id : null, username, hash, email || null, referrer ? referrer.id : null]
      );
      const newUserId = ins.insertId;

      await conn.query("INSERT INTO wallets (user_id, balance) VALUES (?, 0)", [newUserId]);

      if (referrer) {
        const newBonus = settings.referral_new_user_bonus || 0;
        const refBonus = settings.referral_referrer_bonus || 0;

        if (newBonus > 0) {
          await conn.query("UPDATE wallets SET balance = balance + ? WHERE user_id = ?", [newBonus, newUserId]);
        }
        if (refBonus > 0) {
          await conn.query("UPDATE wallets SET balance = balance + ? WHERE user_id = ?", [refBonus, referrer.id]);
        }

        await conn.query(
          `INSERT INTO referrals (referrer_user_id, referred_user_id, bonus_for_referrer, bonus_for_new_user)
           VALUES (?, ?, ?, ?)`,
          [referrer.id, newUserId, refBonus, newBonus]
        );
      }

      await conn.commit();
      conn.release();
      res.json({ success: true });
    } catch (e) {
      await conn.rollback();
      conn.release();
      throw e;
    }
  } catch (err) {
    console.error("signup error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

module.exports = router;
