const express = require("express");
const pool = require("../config/db");
const router = express.Router();

router.get("/dashboard", async (req, res) => {
  const agentId = req.user.id;
  try {
    const [[userAgg]] = await pool.query(
      "SELECT COUNT(*) AS totalUsers FROM users WHERE agent_id = ?",
      [agentId]
    );
    const [[balAgg]] = await pool.query(
      "SELECT balance FROM wallets WHERE user_id = ? LIMIT 1",
      [agentId]
    );
    const [[wdPend]] = await pool.query(
      "SELECT COUNT(*) AS pending FROM withdrawals w JOIN users u ON u.id = w.user_id WHERE u.agent_id = ? AND w.status='pending'",
      [agentId]
    );
    const [[wdApp]] = await pool.query(
      "SELECT COUNT(*) AS approved FROM withdrawals w JOIN users u ON u.id = w.user_id WHERE u.agent_id = ? AND w.status='approved'",
      [agentId]
    );
    const [[profitAgg]] = await pool.query(
      "SELECT COALESCE(SUM(commission_amount),0) AS profit FROM agent_commissions WHERE agent_id = ?",
      [agentId]
    );

    res.json({
      totalUsers: userAgg.totalUsers || 0,
      liveUsers: 0,
      totalBalance: balAgg ? balAgg.balance || 0 : 0,
      pendingWithdrawals: wdPend.pending || 0,
      approvedWithdrawals: wdApp.approved || 0,
      profitLoss: profitAgg.profit || 0
    });
  } catch (err) {
    console.error("agent dashboard error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});



// Agent transfers coins from their wallet to a user under them
router.post("/transfer", async (req, res) => {
  const agentId = req.user.id;
  const { userId, amount } = req.body;
  const targetId = Number(userId);
  const amt = Number(amount);

  if (!targetId || !amt || amt <= 0) {
    return res.status(400).json({ message: "Invalid user or amount" });
  }

  let conn;
  try {
    conn = await pool.getConnection();
    await conn.beginTransaction();

    const [[userRow]] = await conn.query(
      "SELECT id, agent_id FROM users WHERE id = ? FOR UPDATE",
      [targetId]
    );
    if (!userRow || userRow.agent_id !== agentId) {
      await conn.rollback();
      return res.status(400).json({ message: "User is not under this agent" });
    }

    const [[agentWallet]] = await conn.query(
      "SELECT id, balance FROM wallets WHERE user_id = ? FOR UPDATE",
      [agentId]
    );
    if (!agentWallet || agentWallet.balance < amt) {
      await conn.rollback();
      return res.status(400).json({ message: "Insufficient agent balance" });
    }

    await conn.query(
      "UPDATE wallets SET balance = balance - ? WHERE id = ?",
      [amt, agentWallet.id]
    );

    const [[userWallet]] = await conn.query(
      "SELECT id, balance FROM wallets WHERE user_id = ? FOR UPDATE",
      [targetId]
    );
    let targetWalletId = userWallet ? userWallet.id : null;
    if (!targetWalletId) {
      const [ins] = await conn.query(
        "INSERT INTO wallets (user_id, balance) VALUES (?, 0)",
        [targetId]
      );
      targetWalletId = ins.insertId;
    }
    await conn.query(
      "UPDATE wallets SET balance = balance + ? WHERE id = ?",
      [amt, targetWalletId]
    );

    await conn.query(
      `INSERT INTO coin_transactions (from_user_id, to_user_id, amount, type, metadata)
       VALUES (?, ?, ?, 'agent_transfer', JSON_OBJECT('note','agent to user'))`,
      [agentId, targetId, amt]
    );

    await conn.commit();
    res.json({ success: true, agentBalance: agentWallet.balance - amt });
  } catch (err) {
    if (conn) {
      try { await conn.rollback(); } catch {}
    }
    console.error("agent transfer error:", err);
    res.status(500).json({ message: "Internal server error" });
  } finally {
    if (conn) conn.release();
  }
});

// Agent transaction overview: coin tx + commissions
router.get("/transactions", async (req, res) => {
  const agentId = req.user.id;
  try {
    const [tx] = await pool.query(
      `SELECT id, from_user_id, to_user_id, amount, type, metadata, created_at
       FROM coin_transactions
       WHERE from_user_id = ? OR to_user_id = ?
       ORDER BY created_at DESC
       LIMIT 100`,
      [agentId, agentId]
    );
    const [comm] = await pool.query(
      `SELECT id, user_id, game_round_id, commission_amount, created_at
       FROM agent_commissions
       WHERE agent_id = ?
       ORDER BY created_at DESC
       LIMIT 50`,
      [agentId]
    );
    res.json({ transactions: tx, commissions: comm });
  } catch (err) {
    console.error("agent transactions error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

module.exports = router;
