const jwt = require("jsonwebtoken");
const pool = require("../config/db");

async function checkMaintenance(user) {
  try {
    const [rows] = await pool.query("SELECT maintenance_mode, plan_expires_at FROM app_settings WHERE id = 1");
    if (!rows.length) return false;
    let maintenance = rows[0].maintenance_mode;
    const expiresAt = rows[0].plan_expires_at;

    if (expiresAt) {
      const now = new Date();
      const exp = new Date(expiresAt);
      if (exp < now) {
        maintenance = 1;
        if (rows[0].maintenance_mode === 0) {
          try {
            await pool.query("UPDATE app_settings SET maintenance_mode = 1 WHERE id = 1");
          } catch (e) {
            console.warn("Failed to flip maintenance_mode on expiry:", e.message);
          }
        }
      }
    }

    if (!maintenance) return false;
    if (user && user.role === "developer") return false; // developer bypass
    return true;
  } catch (e) {
    console.error("checkMaintenance error:", e.message);
    return false;
  }
}

async function checkSession(jti, userId) {
  try {
    if (!jti || !userId) return false;
    const [rows] = await pool.query(
      "SELECT is_active FROM sessions WHERE token_jti = ? AND user_id = ? LIMIT 1",
      [jti, userId]
    );
    if (!rows.length) return false;
    return rows[0].is_active === 1;
  } catch (e) {
    console.error("checkSession error:", e.message);
    return false;
  }
}

function verifyToken(req, res, next) {
  const header = req.headers["authorization"];
  if (!header) return res.status(401).json({ message: "Missing Authorization header" });

  const [scheme, token] = header.split(" ");
  if (scheme !== "Bearer" || !token) {
    return res.status(401).json({ message: "Invalid Authorization format" });
  }

  try {
    const payload = jwt.verify(token, process.env.JWT_SECRET || "super-secret-change-me");
    req.user = {
      id: payload.id,
      role: payload.role,
      username: payload.username,
      jti: payload.jti || null
    };
  } catch (err) {
    return res.status(401).json({ message: "Invalid or expired token" });
  }

  checkSession(req.user.jti, req.user.id)
    .then(async (ok) => {
      if (!ok) {
        return res.status(401).json({ message: "Session expired or logged out from another device" });
      }
      const isMaint = await checkMaintenance(req.user);
      if (isMaint) {
        return res.status(503).json({ message: "Under maintenance" });
      }
      next();
    })
    .catch(() => {
      return res.status(500).json({ message: "Internal server error" });
    });
}

function requireRole(...roles) {
  return (req, res, next) => {
    if (!req.user || !roles.includes(req.user.role)) {
      return res.status(403).json({ message: "Forbidden" });
    }
    next();
  };
}

module.exports = { verifyToken, requireRole };
