// === V10: OTP LOGIN + FORGOT PASSWORD HELPERS ===
//
// Paste this near the top of backend/src/routes/auth.js
// (after you have 'const pool = require("../config/db");' and 'const router = express.Router();')

const crypto = require("crypto");

function generateOtpCode() {
  return ("" + Math.floor(100000 + Math.random() * 900000));
}

// Request OTP (for login)
router.post("/request-otp", async (req, res) => {
  const { username } = req.body;
  if (!username) return res.status(400).json({ message: "Username required" });

  try {
    const [[user]] = await pool.query(
      "SELECT id FROM users WHERE username = ?",
      [username]
    );
    if (!user) return res.status(404).json({ message: "User not found" });

    const code = generateOtpCode();
    const expiresAt = new Date(Date.now() + 10 * 60 * 1000); // 10 minutes

    await pool.query(
      "INSERT INTO otp_codes (user_id, code, purpose, expires_at) VALUES (?, ?, 'login', ?)",
      [user.id, code, expiresAt]
    );

    // NOTE: In real deployment you would send this via email / SMS.
    // For now we return the code so you can test quickly.
    res.json({ success: true, code, info: "For production, send this via email/SMS instead of returning it." });
  } catch (err) {
    console.error("request-otp error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// Login via OTP (no password)
router.post("/login-otp", async (req, res) => {
  const { username, code } = req.body;
  if (!username || !code) {
    return res.status(400).json({ message: "Username and code required" });
  }

  try {
    const [[user]] = await pool.query(
      "SELECT * FROM users WHERE username = ?",
      [username]
    );
    if (!user) return res.status(404).json({ message: "User not found" });

    const [[otp]] = await pool.query(
      `SELECT * FROM otp_codes
       WHERE user_id = ? AND code = ? AND purpose = 'login' AND used = 0
         AND expires_at > NOW()
       ORDER BY created_at DESC
       LIMIT 1`,
      [user.id, code]
    );

    if (!otp) {
      return res.status(400).json({ message: "Invalid or expired code" });
    }

    // mark used
    await pool.query("UPDATE otp_codes SET used = 1 WHERE id = ?", [otp.id]);

    // reuse your existing login success logic:
    // - create JWT
    // - insert into sessions table (single active session)
    // For example, if your /login route uses a helper createSession(user, req, res)
    // you can call that here as well.

    // Example skeleton (adapt to your existing code):
    const jwt = require("jsonwebtoken");
    const token = jwt.sign(
      { id: user.id, role: user.role_id }, // adapt if you use a different payload
      process.env.JWT_SECRET,
      { expiresIn: "12h" }
    );

    // NOTE: This is minimal; in your real auth.js you probably already record sessions.
    res.json({
      token,
      user: {
        id: user.id,
        username: user.username,
        role: user.role // or map from role_id
      }
    });
  } catch (err) {
    console.error("login-otp error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

// Forgot password via OTP (request + reset)
router.post("/forgot-password", async (req, res) => {
  const { username } = req.body;
  if (!username) return res.status(400).json({ message: "Username required" });

  try {
    const [[user]] = await pool.query(
      "SELECT id FROM users WHERE username = ?",
      [username]
    );
    if (!user) return res.status(404).json({ message: "User not found" });

    const code = generateOtpCode();
    const expiresAt = new Date(Date.now() + 15 * 60 * 1000); // 15 min

    await pool.query(
      "INSERT INTO otp_codes (user_id, code, purpose, expires_at) VALUES (?, ?, 'reset', ?)",
      [user.id, code, expiresAt]
    );

    res.json({ success: true, code, info: "For production, send via email/SMS." });
  } catch (err) {
    console.error("forgot-password error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});

router.post("/reset-password-otp", async (req, res) => {
  const { username, code, newPassword } = req.body;
  if (!username || !code || !newPassword) {
    return res.status(400).json({ message: "All fields are required" });
  }

  try {
    const [[user]] = await pool.query(
      "SELECT * FROM users WHERE username = ?",
      [username]
    );
    if (!user) return res.status(404).json({ message: "User not found" });

    const [[otp]] = await pool.query(
      `SELECT * FROM otp_codes
       WHERE user_id = ? AND code = ? AND purpose = 'reset' AND used = 0
         AND expires_at > NOW()
       ORDER BY created_at DESC
       LIMIT 1`,
      [user.id, code]
    );
    if (!otp) return res.status(400).json({ message: "Invalid or expired code" });

    await pool.query("UPDATE otp_codes SET used = 1 WHERE id = ?", [otp.id]);

    const bcrypt = require("bcryptjs");
    const hash = await bcrypt.hash(newPassword, 10);

    await pool.query(
      "UPDATE users SET password_hash = ? WHERE id = ?",
      [hash, user.id]
    );

    res.json({ success: true });
  } catch (err) {
    console.error("reset-password-otp error:", err);
    res.status(500).json({ message: "Internal server error" });
  }
});
